 /*****************************************************************************

    Javascript simulation of the Anita 811 electronic calculator
    Copyright (c) 2003-2004 Simon Southwell and Hugh Steers

    simon_southwell@bigfoot.com, hugh@voidware.com
    11th November 2003, Bristol and London, UK

    $Id: anita811.js,v 1.24 2004/08/26 17:03:18 Simon Exp $

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Anita 811 images in this simulator have been used with kind permission
    of Hugh Steers (www.voidware.com) Copyright (c) 2003 Hugh Steers.

    Anita 810/Adler 81 images in this simulator have been used with kind 
    permission of Simon Southwell Copyright (c) 2003-2004 Simon Southwell.

    Triump 81 images in this simulator have been used with kind 
    permission of Tony Thimet Copyright (c) 2004 Tony Thimet.

    Although original to the authors, this script is based in concept 
    on Neil Fraser's (neil@vv.carleton.ca) HP-35 simulator. 
    See www.hpmuseum.org.

 *****************************************************************************/

var title  = document.title;
var is_810 = title.indexOf(" 810") != -1;
var is_201 = title.indexOf(" 201") != -1;
var is_104 = title.indexOf(" 104") != -1;
var is_netscape = (navigator.appName=="Netscape") ? 1 : 0;

// Image directories
var leddir = is_201 ? "../common/rockwell_anita/images/" : "../common/anita/images/";
var switchdir = "images/"

// Preload dynamic images
led0 = new Image(14, 15); led0.src=leddir+"led0.jpg";
led1 = new Image(14, 15); led1.src=leddir+"led1.jpg";
led2 = new Image(14, 15); led2.src=leddir+"led2.jpg";
led3 = new Image(14, 15); led3.src=leddir+"led3.jpg";
led4 = new Image(14, 15); led4.src=leddir+"led4.jpg";
led5 = new Image(14, 15); led5.src=leddir+"led5.jpg";
led6 = new Image(14, 15); led6.src=leddir+"led6.jpg";
led7 = new Image(14, 15); led7.src=leddir+"led7.jpg";
led8 = new Image(14, 15); led8.src=leddir+"led8.jpg";
led9 = new Image(14, 15); led9.src=leddir+"led9.jpg";

led0dot = new Image(14, 15); led0dot.src=leddir+"led0dot.jpg";
led1dot = new Image(14, 15); led1dot.src=leddir+"led1dot.jpg";
led2dot = new Image(14, 15); led2dot.src=leddir+"led2dot.jpg";
led3dot = new Image(14, 15); led3dot.src=leddir+"led3dot.jpg";
led4dot = new Image(14, 15); led4dot.src=leddir+"led4dot.jpg";
led5dot = new Image(14, 15); led5dot.src=leddir+"led5dot.jpg";
led6dot = new Image(14, 15); led6dot.src=leddir+"led6dot.jpg";
led7dot = new Image(14, 15); led7dot.src=leddir+"led7dot.jpg";
led8dot = new Image(14, 15); led8dot.src=leddir+"led8dot.jpg";
led9dot = new Image(14, 15); led9dot.src=leddir+"led9dot.jpg";

ledoff      = new Image(14, 15); ledoff.src=leddir+"ledoff.jpg";
ledoffdot   = new Image(14, 15); ledoffdot.src=leddir+"ledoffdot.jpg";
ledminus    = new Image(14, 15); ledminus.src=leddir+"ledminus.jpg";
ledminusdot = new Image(14, 15); ledminusdot.src=leddir+"ledminusdot.jpg";
ledminusoff = new Image(14, 15); ledminusoff.src=leddir+"ledminusoff.jpg";
ledc        = new Image(14, 15); ledc.src=leddir+"ledc.jpg";
lede        = new Image(14, 15); lede.src=leddir+"lede.jpg";
ledcdot     = new Image(14, 15); ledcdot.src=leddir+"ledcdot.jpg";
lededot     = new Image(14, 15); lededot.src=leddir+"lededot.jpg";

switchon  = new Image(35, 17); switchon.src=switchdir+"switchon.jpg";
switchoff = new Image(35, 17); switchoff.src=switchdir+"switchoff.jpg";

// State of the switches
var batt_on = false;
var mem_on  = is_104 ? true : false;
var rnd_on  = is_104 ? true : false;

// Display value string
var disp = "0.";

// Main calculator registers 
var x;                          // accumulator
var y;                          // input register
var m;                          // memory register
var k;                          // Constant

// Global operational state
var is_new_num;                 // Flags start of a new number entry
var is_decimal;                 // Flags that the decimal point has been pressed
var x_is_clear;                 // Flags that the x register has not been loaded since a reset
var last_op;                    // Stores pending multiply or divide operation
var flow_err;                   // Flags over/underflow
var first_num;                  // Flags first number in multiply/divide chain
var last_key;                   // Value of the last input key pressed
var curr_key;                   // Current active key
var ip_dp;                      // Number of entered decimal places at input
var x_dp;                       // Current number of decimal places in x reg
var last_sym;

// Flag if script called from Anita 810 skin by testing HTML document title
var is_810 = document.title == "Anita 810 Calculator";
var valid_keys = is_810 ? ".0123456789%=+-*/CoR" : 
                 is_104 ? ".0123456789=+-*/Corc" :
                          ".0123456789%=+-*/CoRrcm";

// Constant declarations (don't use 'const' which causes compatibility issues)
var MAXDISPLEN  = 10;           // Max length of disp string
var MAXDISPLENP = 9;            // Max length of disp string when positive
var MINDISPLENO = 2;            // Min disp length (when on)
var MAXNUM      = 99999999;     // Minimum number displayable
var MINNUM      = -99999999;    // Maximum number displayable
var RNDFACTOR   = 100000000;    // Rounding factor
var RNDPLACES   = 2;            // Number of rounding places
var NOMATCH     = -1;           // Match fail return value
var RUNTEST     = 0x74;

// ---------------------------------------------------------
// Initialise main state
//
function initialise () {
    disp = "0.";
    x = 0;
    y = 0;
    x_dp = 0;
    x_is_clear = true;
    is_new_num = true;
    is_decimal = false;
    last_op = '';
    last_sym = '';
    last_key = '';
    flow_err = false;
    first_num = true;
}

// ---------------------------------------------------------
// Put display in error (i.e. script bug) state
//
function display_error(type) {
    document.d7.src = led8dot.src;
    document.d6.src = led8dot.src;
    document.d5.src = led8dot.src;
    document.d4.src = led8dot.src;
    document.d3.src = led8dot.src;
    document.d2.src = led8dot.src;
    document.d1.src = led8dot.src;
    document.d0.src = led8dot.src;
}

// ---------------------------------------------------------
// Convert display number to a real value
//
function disp_to_num(dsp) {

    var val;

    val = parseFloat(dsp);
    if (isNaN(val))
        display_error();

    return val;
}

// ---------------------------------------------------------
// Pad str with zeros to at least p decimal places
//
function pad_with_zeros (str, p)
{
    var newstr = str;
    var dps;

    if (is_104)
       return newstr;

    dps = newstr.length - newstr.indexOf('.') - 1;

    while (dps < p && newstr.length < ((newstr.charAt(0) == '-') ? MAXDISPLEN : MAXDISPLENP)) {
        newstr = newstr + '0';
        dps = newstr.length - newstr.indexOf('.') - 1;
    }

    return newstr;
}

// ---------------------------------------------------------

function num_to_string(n) {

    var s = n.toString();
    var i;

    i = s.indexOf('e');
    if (i != NOMATCH) {
        tmp = s.substring(i+1, s.length);
        exp = tmp.valueOf();
	if (exp < -7)
	    s = "0.";
        else {
	    if (n > 0)
	        s = "0.000000" + s.charAt(0);
	    else
	        s = "-0.000000" + s.charAt(1);
        }
    }
    
    return s;
}

// ---------------------------------------------------------
// Convert a real number to a display string
//
// Imported globals : rnd_on, x_dp, last_op, last_key, curr_key
// Exported globals : flow_err

function num_to_disp(numval) {

    var num = numval;
    var is_neg;
    var dstr;

    // Try and correct rounding errors
    num = Math.round(num * RNDFACTOR)/RNDFACTOR;

    if (num > -1e-7 && num < 1e-7)
        num = 0;

    // Flag an overflow/underflow
    flow_err = (num < MINNUM || num > MAXNUM);

    // If divide by 0 overflow, reset the number
    if (num == Infinity) 
        num = 0;

    // Convert number to string
    dstr = num_to_string(num);

    // Append a '.' if none exists
    if (dstr.indexOf('.') == NOMATCH)
        dstr = dstr+'.';

    is_neg = (dstr.charAt(0) == '-');

    // Truncate decimal numbers
    if (dstr.indexOf('.') != dstr.length-1) {

        dstr = dstr.substring(0, is_neg ? MAXDISPLEN : MAXDISPLENP);

        // Remove any trailing zeros
        while (dstr.charAt(dstr.length-1) == '0') 
            dstr = dstr.substring(0, dstr.length-1);

    // Truncate overflow
    } else if (flow_err) 
        dstr = dstr.substring(0, (is_neg ? MAXDISPLEN : MAXDISPLENP) - 1);
      
    // Append a '.' if just truncated it
    if (dstr.indexOf('.') == NOMATCH)
        dstr = dstr+'.';

    // For add/subtract pad the dstrlay to have the x register's
    // number of decimal places 
    if ("*/+-".indexOf(curr_key) != NOMATCH)
        dstr = pad_with_zeros(dstr, x_dp);

    // If rounding, and mult/div type operation, ensure
    // two decimal places (except after a "/=" sequence!)
    else if (rnd_on && last_op != '' && !(last_key == '/' && curr_key == '=')) 
        dstr = pad_with_zeros(dstr, RNDPLACES);

    return dstr;

}

// ---------------------------------------------------------
// Update display graphics from disp value
//
// Valid display string will have between 1 and 8 digits
// an optional '-', and a single '.' (which is at the end for integers).
// All other characters are displayed as 'off'
//
function update_display(dsp) {

    var disp_array = new Array();
    var dot_active = false;             // True when processing decimal digits
    var is_negative;

    is_negative = dsp.charAt(0) == '-';

    // Check validity of input 
    if ((dsp.indexOf('-') > 0) ||
        (dsp != '' && ((dsp.indexOf('.') == NOMATCH) || (dsp.length < MINDISPLENO)))) {
        display_error();
        return;
    }

    // Minus digit off (or overflow) by default
    if (is_201) {
        disp_array[8] = ledminusoff.src;
        disp_array[9] = flow_err ? ledminus.src : ledminusoff.src;
    } else
        disp_array[8] = (is_104 && m != 0 && batt_on) ? 
	                     (flow_err ? ledcdot.src : ledoffdot.src) :
	                     (flow_err ? ledc.src : ledoff.src);

    // Display is right justified
    idx = dsp.length;
    leds = 8;
    while (leds >= 0) {
        idx--;
        digit = dsp.charAt(idx);
        if (digit == '.') {
            dot_active = true;
        } else if (digit == '-') {
	    if (is_201) 
                disp_array[8] = ledminus.src;
            else
                disp_array[8] = (is_104 && m != 0) ?
		                  (flow_err ? lededot.src : ledminusdot.src) :
		                  (flow_err ? lede.src : ledminus.src);
        } else if (digit && "0123456789".indexOf(digit) != NOMATCH) {
            leds--;
            if (dot_active) 
                disp_array[leds] = leddir+"led"+digit+"dot.jpg";
            else
                disp_array[leds] = leddir+"led"+digit+".jpg";
            dot_active = false;
        } else {
            leds--;
            disp_array[leds] = ledoff.src;
        }
    }

    // Update display
    document.d7.src = disp_array[0];
    document.d6.src = disp_array[1];
    document.d5.src = disp_array[2];
    document.d4.src = disp_array[3];
    document.d3.src = disp_array[4];
    document.d2.src = disp_array[5];
    document.d1.src = disp_array[6];
    document.d0.src = disp_array[7];
    document.dm.src = disp_array[8];
    if (is_201)
        document.de.src = disp_array[9];
        
}

// ---------------------------------------------------------
// Process operational keys
//
// Imported globals : y, k, x_dp, ip_dp
// Exported globals : y, x, x_dp, x_is_clear, disp

function reduce(Key, percent)
{
    var xstr;

    if (Key == '+') 
       x += y;
    else if (Key == '-') 
       x -= y;
    else if (Key == '*') 
       x = k * (y / (percent ? 100 : 1));
    else if (Key == '/') {
       x = y / (k / (percent ? 100 : 1));
    }

    x_dp = (x_dp < ip_dp) ? ip_dp : x_dp;

    // Convert x to disp string
    disp = num_to_disp(x);

    // If divide by 0 overflow, clear out the registers
    if (x == Infinity || x == -Infinity) {
        disp = "0.";
        y = 0;
        x = 0;
        x_dp = 0;
    } else 
        // There is now something in the x register
        x_is_clear = false;
}

// ---------------------------------------------------------
// Callback for processing key presses
//
function key_pressed(key) {

    curr_key = key;

    // Not a valid key, error and not the clear key or not switched on! 
    if ((flow_err && key != 'C') || !batt_on)
        return; 

    // Process a number key. Process 0 only if other numbers entered
    else if (((key && "123456789".indexOf(key)) != NOMATCH) || 
              (key == '0' && (!(disp.length == 2 && disp.charAt(0) == '0') || 
               is_decimal == true))) {

        // Don't process key if input register (display) full
        if (!is_new_num && disp.length == MAXDISPLENP)
            return;

        // If decimal active, simply append number
        if (is_decimal) {
            disp = disp+key;
            ip_dp = ip_dp + 1;
        // If decimal not active keep decimal at the end
        } else {
            ip_dp = 0;
            if (is_new_num) 
                disp = key+'.';
            else 
                disp = disp.substring(0, disp.length-1)+key+'.';
        }

        // Number not new after first key press
        is_new_num = false;

        y = disp_to_num(disp);
    }
    else if (key == '.') {

        if (!is_decimal)
            ip_dp = 0;

        // If '.' is the first key of a new number, imply '0'
        // already pressed.
        if (is_new_num) {
            disp = "0.";
            is_new_num = false;
        } 

        is_decimal = true;
        y = disp_to_num(disp);
    }
    else if ("*/".indexOf(key) != NOMATCH) {

        // Treat accumulator as input for chained ops after an equals or add/subtract
        if ("=+-".indexOf(last_key) != NOMATCH) {
           y = x;
        }

        // If first number in a mult/div chain, simply copy to x register
        else if (first_num) {
           x = y;
           x_is_clear = false;
           x_dp = ip_dp;

        // Perform op
        } else {
            // If chained divide (except mem recall), set constant to y and
            // treat x as an input
            if (last_op == '/' && last_key != 'r') {
                k = y;
                y = x;
            }

           reduce(last_op, false);
        }

        // Constant becomes x under all circumstances
        k = x;

        // Display x
        disp = num_to_disp(x);

        is_new_num = true;
        is_decimal = false;
        last_op = key;
        last_sym = key;
        first_num = false;
    }
    else if ("+-".indexOf(key) != NOMATCH) { 

        // If last key '%=' swap x into y and constant into x
        if (last_key == '%') {
            y = x;
            x = k;
            x_is_clear = false;
        }

	// New chain started if pressed equals, then entered a new number
	if (last_sym == '=' && !is_new_num) {
	    x_is_clear = true;
	    x = 0;
	}

        // Perform op unless last key is '=' where x is treated as an input
        if (last_key != '=')
            reduce(key, false);

        is_new_num = true;
        is_decimal = false;
        last_op = '';
        last_sym = key;
        first_num = true;
    }
    else if ("=%".indexOf(key) != NOMATCH) {

        // If chained divide (except mem recall), set constant to y and
        // treat x as an input
        if (!first_num && last_op == '/' && last_key != 'r') {
            k = y;
            y = x;
        }

        // After an equals, treat x as an input
        if (last_key == '=') {
	    k = y;
            y = x;
	}

        // If nothing in the x register, operate on input (y) reg; rounding
        // and adding to memory as necessary. No further processing
        if (x_is_clear) {
            if (rnd_on) {
                y = Math.round(y*100)/100;
                disp = num_to_disp(y);
            }
            if (mem_on) 
               m = m + y;

            return;
        } 
        // If equals pressed after entering a number, which isn't a multiply/divide 
        // chain, make the input the new accumulator value.
        else if (first_num && "0123456789".indexOf(last_key) != NOMATCH) {
            x = y;
            x_is_clear = false;
        }

        // Perform op, flagging if '%='
        reduce(last_op, (key == '%'));

        // Round result if switch active
        if (rnd_on && !flow_err) {
           x = Math.round(x*100)/100;
           disp = num_to_disp(x);
        }

        // Store result in memory if switch active
        if (mem_on) 
            m = m + x;


        first_num = true;
        is_new_num = true;
        is_decimal = false;
        last_sym = key;
        x_dp = 0;
    }
    // Process memory stuff if not an 810
    if (key == 'c' && !is_810) {

        m = 0;

        // Start a new number entry
        is_new_num = true;
        is_decimal = false;
        last_op = '';
    }
    else if (key == 'r' && !is_810) {

        // For divide, operator is k reg, so restore to that,
        // else restore to y (input) register
        if (last_key == '/')
            k = m;
        else
            y = m;
    
        disp = num_to_disp(m);

        // Start a new number entry
        is_new_num = true;
        is_decimal = false;
    }
    // Process clear
    else if (key == 'C') {

        // If an error outstanding, clear the error and
        // rescale the x register.
        if (flow_err) {
            flow_err = false;
            while (x < MINNUM || x > MAXNUM)
                x = x / 10;
            k = x;
        }
        // If entering a number, simply reset the input register
        // back to 0
        else if (!is_new_num) {
           y = 0;
           disp = num_to_disp(y);
           is_new_num = true;
           is_decimal = false;
        }
        // A full clear
        else
           initialise();
    }

    // Update display
    update_display(disp);

    last_key = key;
}

// ---------------------------------------------------------
// Callback when battery switch is changed
//
function sw_batt() {
    batt_on = !batt_on
    if(is_201)
        document.batt_img.src = batt_on ? switchoff.src : switchon.src;
    else
        document.batt_img.src = batt_on ? switchon.src : switchoff.src;
    if (batt_on) {
        disp = "0.";
        initialise();
        m = 0;
    } else {
        disp = "";
        flow_err = false;
    }

    update_display(disp);
}

// ---------------------------------------------------------
// Callback when rounding switch is changed
//
function sw_round() {
    rnd_on = !rnd_on;
    document.rnd_img.src = rnd_on ? switchon.src : switchoff.src;
}

// ---------------------------------------------------------
// Callback when mem switch is changed
//
function sw_mem() {
    mem_on = !mem_on;
    document.mem_img.src = mem_on ? switchon.src : switchoff.src;
}
// ---------------------------------------------------------
// Keyboard pressed callback
//
function keyboard(e) {

    var ky, code;

    code = is_netscape ? e.which : e.keyCode;

    // Execute test on ^T
    if (code == RUNTEST) {
        test();                 // Uncomment to enable testing
        return;
    }

    // Extract key value from event and convert to character
    ky = String.fromCharCode(code);

    // Ignore all key presses that are not mapped
    if (valid_keys.indexOf(ky) == NOMATCH) 
        return; 

    // Catch keys for changing switches
    if (ky == 'o')
        sw_batt();
    else if (ky == 'm' && !is_810)
        sw_mem();
    else if (ky == 'R')
        sw_round();
    // All other key presses passed on
    else
        key_pressed(ky);
}

// ---------------------------------------------------------
// Logo clicked callback
//
function logo() {

  // Display and 'about' message
  alert("A JavaScript Anita 811/810 simulator\n" +
        "by Simon Southwell (simon_southwell@bigfoot.com)\n" +
        "and Hugh Steers (hugh@voidware.com)\n" +
        "January 2004, Bristol and London, UK.\n\n" +
        "RELEASE_2004_08_26");
}

// Test data string format is:
//
// <input key sequence> : <expected displayed result> @ ....
//
// For the Anita 811 manual example tests, when the rounding switch (R) 
// or memory switch (m) are activated for a set of sequences, they must be 
// deactivated before moving on to the next sequence set.
// Spaces in the input sequence are ignored (but don't use tabs).

testdata = "oo:0@" +                                    // initialise
"C 18.5 + 7.9 + 33 +                : 59.4 @" +         // test 1
"R C 13.56 + 22.8462 -              : -9.2862 @" +
"                    =             R: -9.29 @" +
"  C 9 + 3.4 + + +                  : 19.2 @" +
"  15.123 * 9.5  =                  : 143.6685 @" +
"R 15.123 * 9.5  =                 R: 143.67 @" +
"  159 / 14 =                       : 11.357142 @" +
"R 159 / 14 =                      R: 11.36 @" +
"1.11 * 2.22 * 789 =                : 1944.2538 @" +
"R 1.11 * 2.22 * 789 =             R: 1944.25 @" +      // test 10
"  12 * 7.95 / 18.4 =               : 5.1847826 @" +
"  123456 * 456789 =                : C56393342 @" +
"                  C                : 56393342 @" +
"                  C                : 0 @" +
"  17 * 9.2 = + 14.3 +              : 170.7 @" +
"R 18.13 * 3.2 =                    : 58.02 @" +
"          7.15 =                   : 129.63 @" +
"          5.83 =                  R: 105.70 @" +
"R 21.19 / 4.2573 =                 : 4.98 @" +
"          49.08 =                  : 11.53 @" +        // test 20
"          18.75 =                 R: 4.40 @" +
"  1.07 * =                         : 1.1449 @" +
"  1.07 * = =                       : 1.225043 @" +
"  1.07 * = * =                     : 1.310796 @" +
"  1.07 * = * = * =                 : 1.7181861 @" +
"R 125 * 15 %                       : 18.75 @" +
"           +                      R: 143.75 @" +
"R 189.5 * 19.2 %                   : 36.38 @" +
"               -                  R: 153.12 @" +
"R 35 / 289 %                      R: 12.11 @" +        // test 30
"R m C c 240 =                      : 240 @" +
"            -                      : -240 @" +
"        265 +                      : 25 @" +
"            / r %                mR: 10.42 @" +
"R m C c 265 =                      : 265 @" +
"            -                      : -265 @" +
"        240 +                      : -25 @" +
"            / r %                mR: -9.43 @" +
"R m c 12.8 * 9.3 =                 : 119.04 @" +
"      7.6 * 4.8 =                  : 36.48 @" +        // test 40
"      2.5 * 6.5 =                  : 16.25 @" +
"                r                mR: 171.77 @" +
"R m c 3 * 2.7 * .86 =              : 6.97 @" +
"      4 * 8.2 * .71 =              : 23.29 @" +
"                    r            mR: 30.26 @" +
"R m c 17.725 * 13.5 =              : 239.29 @" +
"      8.075 * 17.3 =               : 139.70 @" +
"                   r               : 378.99 @" +
"            * 15 %                 : 56.85 @" +
"                 -                 : 322.14 @" +       // test 50
"            * 11 %                 : 35.44 @" +
"                 +                 : 357.58 @" +
"            * 3.5 %                : 12.52 @" +
"                 -               mR: 345.06 @" + 
"R C c 2795+8015+6244+1920 +        : 18974 @" +
"  /= m                             : 1 @" +
"  2795 %                           : 14.73 @" +
"  8015 %                           : 42.24 @" +
"  6244 %                           : 32.91 @" +
"  1920 %                           : 10.12 @" +        // test 60
"       r                         mR: 100.00 @" +       // End of Anita 811 manual examples
"C 1 + 3 C                          : 0 @" +
"          2 +                      : 3 @" +
"C 2 + + + +                        : 8 @" +
"C 2 * * * *                        : 16 @" +
"C .001                             : 0.001 @" +
"C 2 + 3 + * 5 =                    : 25 @" + 
"R C 5.555 + 7.777 =                : 7.78 @" +         // Check '=' restarts chain
"    + 4.444 +                     R: 12.224 @" +
"R C 8.789 + .006 =                 : 0.01 @" +         // test 70
"    * 11.78129 =                  R: 0.12 @" +
"C 22 / 0 =                         : C0 @" +           // Divide by 0 and clear
"C                                  : 0 @" +
"  2.3 + 4.8 +                      : 7.1 @" +
"C 50.0001 + 49.9999 +              : 100.0000 @" +     // Check trailing zeros
"C 4.200 + 7 +                      : 11.200 @" +       
"C 4.20 + 7.0000 +                  : 11.2000 @" +      
"C 3.100 + 295141 +                 : 295144.10 @" +
"C 1800.00 *                        : 1800.00 @" +
"  2.00 =                           : 3600 @" +
"C 550 / 1 * 12 =                   : 6600 @" +         // Divide chain
"C 2*4=6+                           : 6 @" +
"      8+                           : 14 @" +
"C 2*3*==                           : 54 @" +

"oo:0"; // last line
// ---------------------------------------------------------
function test() {

    var answer = false;                 // State indicating reading expected answer
    var testnum = 0;                    // test number
    var failures = new Array();         // List of test failures
    var fidx = 0;                       // Failure index
    var testchar = "";                  // Current input character

    // Can't do 811 tests in 810 mode
    if (is_810 || is_104) {
        alert("Unable to run tests in Anita 810 mode.\n" +
              "Switch to an Anita 811 skin.");
        return;
    } else  if (!confirm("Run the self tests?\n(may take some seconds)"))
        return;

    // Ensure calculator is switched on
    if (!batt_on)
        sw_batt();

    // Scan through all test data ...
    for (tidx = 0; tidx < testdata.length; tidx = tidx + 1) {

        // get next input character
        testchar = testdata.charAt(tidx);

        // Non white-space input
        if (testchar != ' ') {
            // If answer delimiter, clear expected answer string and set state
            if (testchar == ':') {
                answer = true;
                expected = "";
            }
            // If test delimiter, check answer
            else if (testchar == '@') {
                // Clear state
                answer = false;

                // Get calculator's result
                result = disp;

                // For an over/under flow error, check sign match
                if (flow_err) {
                    if ((expected.charAt(0) == 'C' && result.charAt(0) == '-' ) ||
                        (expected.charAt(0) == 'E' && result.charAt(0) == '-' ))
                        failures[fidx++] = testnum;

                    // Make answer and result sign independent before checking
                    // actual value
                    expected = expected.substring(1, expected.length);
                    if (result.charAt(0) == '-')
                        result = result.substring(1, result.length);
                }

                // Check result matches expected, and log if not
                if (parseFloat(expected) != parseFloat(result)) 
                    failures[fidx++] = testnum;
 
                // Check for equal number of decimal places
                if (expected.indexOf('.') == NOMATCH)
                    expected = expected + '.';
                if ((expected.length - expected.indexOf('.')) != (result.length - result.indexOf('.')))
                    failures[fidx++] = testnum;

                testnum++;
            }
            // Whilst in answer state, append input to expected variable
            else if (answer == true)
                expected = expected + testchar;
            // Input key sequences
            else {
                // Abort if an invalid character found in the test data
                if (valid_keys.indexOf(testchar) == NOMATCH) {
                    alert("Invalid input character ("+testchar+") in test sequence\n"+
                          "at line "+testnum);
                    return;
                }

                // Process switch keys here
                if (testchar == 'o')
                    sw_batt();
                else if (testchar == 'm')
                    sw_mem();
                else if (testchar == 'R')
                    sw_round();
                // All other test keys passed on
                else
                    key_pressed(testchar);
            }
        }
    }

    // Clear testchar and reuse for failure message
    testchar = "";

    // Add all failure test numbers to string
    for (tidx = 0; tidx < fidx; tidx++) 
        testchar = testchar + " " + failures[tidx];

    // Display test results
    if (fidx > 0)
        alert (fidx+" failures at lines:"+testchar);
    else
        alert ("All "+ (testnum-1) + " tests pass");

}
