 /*****************************************************************************

    Javascript simulation of the Anita 841 electronic calculators
    Copyright (c) 2004-2006 Simon Southwell

    simon_southwell@bigfoot.com
    31st March 2004, Bristol, UK

    $Id: anita841.js,v 1.12 2004/11/11 18:34:48 Simon Exp $

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Anita 841 images in this simulator have been used with kind permission
    of Hugh Steers (www.voidware.com) Copyright (c) 2003 Hugh Steers.

 *****************************************************************************/

var is_netscape = (navigator.appName=="Netscape") ? 1 : 0;

var title  = document.title;
var is_8041 = title.indexOf(" 8041") != -1;

leddir=is_8041 ? "../common/ficord/images/" : "../common/anita/images/"
var switchdir="images/"

// Preload dynamic images
led0 = new Image(14, 15); led0.src=leddir+"led0.jpg";
led1 = new Image(14, 15); led1.src=leddir+"led1.jpg";
led2 = new Image(14, 15); led2.src=leddir+"led2.jpg";
led3 = new Image(14, 15); led3.src=leddir+"led3.jpg";
led4 = new Image(14, 15); led4.src=leddir+"led4.jpg";
led5 = new Image(14, 15); led5.src=leddir+"led5.jpg";
led6 = new Image(14, 15); led6.src=leddir+"led6.jpg";
led7 = new Image(14, 15); led7.src=leddir+"led7.jpg";
led8 = new Image(14, 15); led8.src=leddir+"led8.jpg";
led9 = new Image(14, 15); led9.src=leddir+"led9.jpg";

led0dot = new Image(14, 15); led0dot.src=leddir+"led0dot.jpg";
led1dot = new Image(14, 15); led1dot.src=leddir+"led1dot.jpg";
led2dot = new Image(14, 15); led2dot.src=leddir+"led2dot.jpg";
led3dot = new Image(14, 15); led3dot.src=leddir+"led3dot.jpg";
led4dot = new Image(14, 15); led4dot.src=leddir+"led4dot.jpg";
led5dot = new Image(14, 15); led5dot.src=leddir+"led5dot.jpg";
led6dot = new Image(14, 15); led6dot.src=leddir+"led6dot.jpg";
led7dot = new Image(14, 15); led7dot.src=leddir+"led7dot.jpg";
led8dot = new Image(14, 15); led8dot.src=leddir+"led8dot.jpg";
led9dot = new Image(14, 15); led9dot.src=leddir+"led9dot.jpg";

ledoff   = new Image(14, 15); ledoff.src=leddir+"ledoff.jpg";
ledminus = new Image(14, 15); ledminus.src=leddir+"ledminus.jpg";
lede     = new Image(14, 15); lede.src=leddir+"ledbr.jpg";

switchon  = new Image(35, 17); switchon.src=switchdir+"switchon.jpg";
switchoff = new Image(35, 17); switchoff.src=switchdir+"switchoff.jpg";

// State of the switches
var batt_on     = false;
var rnd_on      = false;
var deg_on      = false;

var x=0;                     // Accumulator
var y=0;                     // Input/display register
var m=0;                     // Memory register
var k=0;                     // Constant register
var k_op = "";               // Constant operation

var disp            = "0.";  // Display in string format
var is_new_num      = true;  // New number to be input
var is_decimal      = false; // Input has decimal active
var last_op         = "";    // Last operation key pressed
var error           = false; // Overflow/undeflow error
var infinity        = false; // Infinity (divide by 0)
var func_active     = false; // Function key ('f') pressed
var arc_active      = false; // Arc key pressed
var disp_is_x_not_y = true;  // Flag that X reg is dsplayed, not y

// Valid key inputs
var validkeys = "0123456789.+-*/=fiscod";

var testing = false;
var busy = false;
var delay = 0;

// Constant values
var CTL_T   = 0x14;                           // ASCII code for ^T
var NOMATCH = -1;                             // Match fail return value
var MINVAL  = -99999999;                      // Minimum value
var MAXVAL  =  99999999;                      // Maximum value
var RNDFACTOR = 100000000;                    // Rounding factor
var RADCONV = 57.2957795131;                  // Degrees to radians factor
var TRIGDELAY = 1000;                         // Default delay for trig functions

// ---------------------------------------------------------

function update_display() {

    var disp_array = new Array();
    var dot_active = false;
    var dsp=disp;
    var leds=0;
    var idx=dsp.length;

    if (disp.indexOf('.') == NOMATCH) {
        dsp = disp + '.';
        idx++;
    } else {
        while (dsp.length > 9 && dsp.charAt(dsp.length-1) != ".") {
            dsp = dsp.substring(0, dsp.length-1);
	    idx--;
        }
    }

    if (error)
        document.dm.src = lede.src;
    else if (dsp.charAt(0) == '-')
        document.dm.src = ledminus.src;
    else
        document.dm.src = ledoff.src;

    // Fill in the main display array
    while (leds < 8) {
        idx--;
        digit = dsp.charAt(idx);
        if (digit == '.') 
            dot_active = true;
        else if (digit && "0123456789".indexOf(digit) != NOMATCH) {
            if (dot_active) 
                disp_array[leds] = leddir+"led"+digit+"dot.jpg";
            else
                disp_array[leds] = leddir+"led"+digit+".jpg";
            dot_active = false;
            leds++;
        } else {
            disp_array[leds] = ledoff.src;
            leds++;
        }
    }

    // Update display
    document.d7.src = disp_array[7];
    document.d6.src = disp_array[6];
    document.d5.src = disp_array[5];
    document.d4.src = disp_array[4];
    document.d3.src = disp_array[3];
    document.d2.src = disp_array[2];
    document.d1.src = disp_array[1];
    document.d0.src = disp_array[0];
    
    busy = false;
    delay = 0;
}

// ---------------------------------------------------------

function clr_display() {
    document.dm.src = ledoff.src;
    document.d7.src = ledoff.src;
    document.d6.src = ledoff.src;
    document.d5.src = ledoff.src;
    document.d4.src = ledoff.src;
    document.d3.src = ledoff.src;
    document.d2.src = ledoff.src;
    document.d1.src = ledoff.src;
    document.d0.src = ledoff.src;
}

// ---------------------------------------------------------

function initialise() {
   x = 0;
   y = 0;
   k = 0;
   k_op = "";

   disp = "0.";
   is_new_num = true;
   is_decimal = false;
   last_op = "";
   error = false;
   infinity = false;
   func_active = false;
   arc_active = false;
   disp_is_x_not_y = true;
   busy = false;
   delay = 0;

}

// ---------------------------------------------------------

function rnd_to_display(valin, op) {

    var rnd_factor;
    var val;
    var result;

    val = Math.round(valin * RNDFACTOR)/RNDFACTOR;

    result = (val < 0) ? -1 * val : val;

    if (result < 1e-7)
        return 0;

    if ("p*/".indexOf(op) != NOMATCH) {
        if (rnd_on) {
            rnd_factor = 100;
            result = (Math.round(val * rnd_factor))/rnd_factor;
        } else {
            rnd_factor = 10000000;
            while (result >= 10) {
                result = result / 10;
                rnd_factor = rnd_factor / 10;
            }
            result = (Math.floor(val * rnd_factor))/rnd_factor;
        }
    } else {
        result = Math.round(val * 10000000)/10000000;
    }

    return result;
}

// ---------------------------------------------------------

function num_to_string(n) {

    var s = n.toString();
    var i;

    i = s.indexOf('e');
    if (i != NOMATCH) {
        tmp = s.substring(i+1, s.length);
        exp = tmp.valueOf();
        if (exp < -7)
            s = "0.";
        else {
            if (n > 0)
                s = "0.000000" + s.charAt(0);
            else
                s = "-0.000000" + s.charAt(1);
        }
    }
    
    return s;
}

// ---------------------------------------------------------

function reduce (op) {

    if (op == '+')
        x = x + y;
    if (op == '-')
        x = x - y;
    if (op == '*')
        x = x * y;
    if (op == '/') {
        // Trap divide by 0
        if (y == 0) {
            infinity = true;
            error = true;
            x = 0;
            return;
        }  
        x = x / y;
    }
    else if (op == 'p') {
        x = Math.pow(x, y);
	delay = TRIGDELAY;
    }

    // Check for out-of-range, and rescale
    if (x < MINVAL || x > MAXVAL) {
        error = true;
        x = x / 100000000;
    }


    x = rnd_to_display(x, op);

    disp_is_x_not_y = true;
}

// ---------------------------------------------------------
// Callback when battery switch is changed
//
function sw_batt() {
    batt_on = !batt_on
    document.batt_img.src = batt_on ? switchon.src : switchoff.src;
    if (batt_on) {
        initialise();
        m = 0;
    } else {
        disp = "";
        flow_err = false;
    }

    update_display();
}

// ---------------------------------------------------------
// Callback when mem switch is changed
//
function sw_deg() {
    deg_on = !deg_on;
    document.ang_img.src = deg_on ? switchon.src : switchoff.src;
}
// ---------------------------------------------------------

function keyboard(e) {

    var ky;

    code = is_netscape ? e.which : e.keyCode;
    ky = String.fromCharCode(code);

    if (e.keyCode == CTL_T)
        test();
    else if (busy || validkeys.indexOf(ky) == NOMATCH)
        return;
    // Catch keys for changing switches
    else if (ky == 'o')
        sw_batt();
    else if (ky == 'd')
        sw_deg();
    // All other key presses passed on
    else
        key_pressed(ky);
}

// ---------------------------------------------------------
function freduce(op, val) {

    var n = val;
    var tmp;

    // Memory ops
    if ((!is_8041 && op == 's') || (is_8041 && op == 'i')) {
        m = n;
    } else if ((!is_8041 && op == 'i') || (is_8041 && op == 's')) {
        tmp = m;
        m = n;
        n = tmp;
        disp_is_x_not_y = false;
    } else if (op == '=') {
        n = m;
        disp_is_x_not_y = false;
    } else if (".+-".indexOf(op) != NOMATCH) {
        if (op == '.')
            m += n*n;
        else if (op == '+')
            m += n;
        else if (op == '-')
            m -= n;

        if (m < MINVAL || m > MAXVAL) {
            m = m / 100000000;
    }
    // Arithmetic
    } else if (op == '/') {
        if (n == 0)
            error = true;
        else 
            n = 1/n;
        disp = num_to_string(n);

    } else if (op == '9') {
        if (n < 0)
            error = true;
        else 
            n = Math.sqrt(n);

    // Trigonometric
    } else if (op == '1') {
        if (arc_active) {
            n = Math.asin(n);
            if (deg_on)
                n *= RADCONV;
        } else {
            if (deg_on)
                n /= RADCONV;
            n = Math.sin(n);
        }
    } else if (op == '2') {
        if (arc_active) {
            n = Math.acos(n);
            if (deg_on)
                n *= RADCONV;
        } else {
            if (deg_on)
                n /= RADCONV;
            n = Math.cos(n);
        }
    } else if (op == '3') {
        if (arc_active) {
            n = Math.atan(n);
            if (deg_on)
                n *= RADCONV;
        } else {
            if (deg_on)
                n /= RADCONV;
            n = Math.tan(n);
        }

    // Logarithmic
    } else if (op == '4')
        n = Math.exp(n);
    else if (op == '7') {
        if (n == 0)
            error = true;
        else
            n = Math.log(n);
    }
    else if (op == '5') {
        n = Math.pow(10, n);
        // Check for out-of-range, and rescale
        if (n < MINVAL || n > MAXVAL) {
            error = true;
            n = n / 100000000;
        }

   } else if (op == '8') {
        if (n == 0)
            error = true;
        else
            n = Math.log(n)/2.30258509299;
    }

    if (op >= '1' && op < '9')
        delay = TRIGDELAY;

    n = rnd_to_display(n, "*");

    // Convert x to disp string
    disp = num_to_string(n);

    return n;

}
// ---------------------------------------------------------

function key_pressed(key) {

    if ((error && key != 'c') || !batt_on)
        return;

    // Function operations
    if (func_active) {
        if (key == 'c') {
            func_active = false;
            arc_active = false;
        } else if (key == 'f') {
            if (!disp_is_x_not_y) {
                if (disp.charAt(disp.length-1) == '.')
                    disp = disp.substring(0, disp.length-2) + '.';
                else 
                    disp = disp.substring(0, disp.length-1);
                if (disp == ".")
                    disp = "0.";

                if (disp == "0.") {
                    y = x;
                    disp = num_to_string(x);
                } else 
                    y = parseFloat(disp);
            }
        } else if (key == '0') {
            arc_active = true;
        } else if (key == '*') {
            y = 3.1415926;
            disp_is_x_not_y = false;
            disp = num_to_string(y);
            is_new_num = false;
        } else if (key == '6') {

            x = y;
	    disp = num_to_string(rnd_to_display(Math.log(x), 'p'));
	    delay = TRIGDELAY;

            last_op = 'p';

        } else {

            if (disp_is_x_not_y && key != '=') {
                x = freduce(key, x);
                y = x;
            } else {
                y = freduce(key, y);
                disp_is_x_not_y = false;
            }
        }

        // Except on Arc and f, func active is cleared on all keys
        if (key != '0' && key != 'f') {
            func_active = false;
            arc_active = false;
        }

        if (key != 'c' && key != 'f') {
            is_new_num = true;
            is_decimal = false;
        }

    // Numeric input
    } else if ((key && ".0123456789".indexOf(key)) != NOMATCH) {

        disp_is_x_not_y = false;

        if (key == '.') {

            if (is_new_num) {
                disp = "0.";
                is_new_num = false;
            } 
            is_decimal = true;

        } else {
            if (!is_new_num && disp.length == 9)
                return;

            if (is_decimal) {
                disp = disp+key;
            } else if (y == 0 && key == '0') {
                disp = "0.";
                y = 0;
                return;
            } else {
                if (is_new_num) 
                    disp = key+'.';
                else 
                    disp = disp.substring(0, disp.length-1)+key+'.';
            }

            y = parseFloat(disp);

            is_new_num = false;
            func_active = false;
    
        }
    // Operator input
    } else if ("*/+-".indexOf(key) != NOMATCH && !func_active) {

        if (last_op != "") {
            reduce(last_op);
            disp = num_to_string(x);
        } else 
           x = y;

        is_new_num = true;
        is_decimal = false;
        last_op = key;

    // Equals input
    } else if (key == '=' && !func_active) {

        // If an op key pending, save input and op as constant
        if (last_op != "") {
            k = y;
            k_op = last_op;
        // If no pending op, copy input to x, and use constant
        } else {
            x = y;
            y = k;
            last_op = k_op;
        }

        reduce(last_op);
        y = x;
        disp = num_to_string(y);

        last_op = "";
        is_new_num = true;
        is_decimal = false;
        func_active = false;

    // Auxilliary operation input
    } else {
        // Change sign root
        if (key == 's') {
            if (disp_is_x_not_y) {
                x = -1 * x;
                if (x < 0) disp = "-" + disp;
                else if (x != 0) disp = disp.substring(1, disp.length);
            } else {
                y = -1 * y;
                if (y < 0) disp = "-" + disp;
                else if (y != 0) disp = disp.substring(1, disp.length);
            }
        // Interchange X and Y
        } else if (key == 'i') {

            tmp = x;
            x = (last_op == "") ? k : y;
            y = tmp;
            if(disp_is_x_not_y)
                disp = num_to_string(x);
            else
                disp = num_to_string(y);

        // function key 
        } else if (key == 'f') {
            func_active = true;

        // Clear key
        } else if (key == 'c') {
            // Clear error
            if (error) {
                error = false;
                infinity = false;
                y = x;
                disp = num_to_string(y);
            // Clear input
            } else if (!is_new_num) {
                y = 0;
                disp = "0.";
            // Reset all
            } else 
                initialise();

            is_new_num = true;
            is_decimal = false;
            func_active = false;
        }
    }

    if (!testing) {
        busy = true;
        clr_display();
        setTimeout(update_display, delay);
    } else
       update_display();
}

// ---------------------------------------------------------

function logo() {

  // Display an 'about' message
  alert("A JavaScript Anita 841 simulator\n" +
        "by Simon Southwell (simon_southwell@bigfoot.com)\n" +
        "April 2004, Bristol, UK.\n\n" +
        "RELEASE_2006_01_07");
}

// ---------------------------------------------------------

testdata = "oo:0.@" +                                                              // initialise
"7 + 5 =                                                         : 12. @" +
"7 - 5 =                                                         : 2. @" +         // p6-7
"7 * 5 =                                                         : 35. @" +
"7 / 5 =                                                         : 1.4 @" +
"5 * 6 * 7 + 8 / 9 - 10 =                                        : 14.222222 @" +
"2 * 12 =                                                        : 24. @" +        // p8-9
"3 =                                                             : 36. @" +
"5 =                                                             : 60. @" +
"12 / 3 =                                                        : 4. @" +
"18 =                                                            : 6. @" +
"25 =                                                            : 8.3333333 @" +
"4 + 21 =                                                        : 25. @" +        // p10-11
"6 =                                                             : 27. @" +
"12 =                                                            : 33. @" +
"12 - 5 =                                                        : 7. @" +
"6 =                                                             : 1. @" +
"3 =                                                             : -2. @" +
"5 * =                                                           : 25. @" +        // p12-13
"6 s * = =                                                       : -216. @" +
"3 * = * =                                                       : 81. @" +
"2 * = = * = =                                                   : 512. @" +
"2.7 * = = / 6 * f * =                                           : 10.305994 @" +
"1.144 * = * = = * 5896 -                                        : 13216.409 @" +  // p 14-15 : manual gives 13216.413
"      /                                                         : 7320.409 @" +   //           manual gives 7320.413
" * 100 =                                                        : 124.1589 @" +   //           manual gives 124.15897
"3 + 4 / 5 i =                                                   : 0.7142857 @" +  // p16-17
"3 + 4 / 5 i - 1 i =                                             : 0.2857143 @" +
"18 * 15 =                                                       : 270. @" +
"   * .05 s i +                                                  : -13.5 @" +
"               =                                                : 256.5 @" +
"   * .08 i +                                                    : 20.52 @" +
"               =                                                : 277.02 @" +
"c f s 2 * 3 = f +                                               : 6. @" +         // p18-19
"4 * 5 = f +                                                     : 20. @" +
"6 * 7 = f +                                                     : 42. @" +
"        f =                                                     : 68. @" +
"2 + 3 = f s 4 + 5 * f = = f s  6 + 7 * f = =                    : 585. @" +
"3 f 9                                                           : 1.7320508 @" +  // p20-21
"4 * 7 = f 9                                                     : 5.2915026 @" +
"2 / 3 f 9 =                                                     : 1.1547005 @" +
"81 f 9 f 9                                                      : 3. @" +
"7 f / * = - 1 i = f 9                                           : 0.9897433 @" +
"100 f 8                                                         : 2. @" +         // p22-23
"2 f 5                                                           : 100. @" +
"10 f 7                                                          : 2.302585 @" +
"2.302585 f 4                                                    : 9.999999 @" +   // Manual gives 9.999995
".7 s * 1200 * .00001 = f 4                                      : 0.9916351 @" +  // Manual gives 0.991636
"f * f 4                                                         : 23.140691 @" +  // Manual gives 23.14068
"1 f 4 f s f * f 6 f = =                                         : 22.459155 @" +  // Manual gives 22.45919
"5 f 6 2 =                                                       : 25. @" +        // p24-25
"1.05 f 6 20 =                                                   : 2.6532977 @" +  // Manual gives 2.653288
"4 f 6 .5 s =                                                    : 0.5 @" +
"215 / 95 = f 6 1.4 =                                            : 3.137623  @" +  // Manual gives 3.137622
"4 / 11 = f s 3 / 7 = f 6 f = =                                  : 0.7348357 @" +  // Manual gives 0.734836
"3 f 9 f s f 6 f = =                                             : 2.5893998 @" +  // Manual gives 2.589398
"4785 f 6 3 f / =                                                : 16.851058 @" +  // p26-27 Manual gives 16.85106
"15 * 20 = f 6 .1 =                                              : 1.768936 @" +   // Manual gives 1.768935
"23456 / 5896 = f 6 7 f / =                                      : 1.2180662 @" +  // Manual gives 1.218066
"* 100 - =                                                       : 21.80662 @" +   // Manual gives 21.8066
"5896 / 23456 = f 6 7 f / =                                      : 0.8209734 @" +  // Manual gives 0.820974
"* 100 - =                                                       : -17.90266 @" +  // Manual gives -17.9026
"2 f 7 / 20 i = f s 5 f 7 * f = =                                : 46.438566 @" +  // p28-29 Manual gives 46.4385
".5 f 7 / .1 i = f s 1000000 f / f 7 * f = =                     : 1.9931568 @" +  // Manual gives 1.99315
"d 30 f 1                                                        : 0.5 @" +        // p30-31
"270 f 2                                                         : 0. @" +
"135 f 3                                                         : -1. @" +
"120 f 1 f /                                                     : 1.1547005 @" +  // Manual gives 1.1546997
"45 f 2 f /                                                      : 1.4142137 @" +  // Manual gives 1.4142131
"150 f 3 f /                                                    d: -1.7320508 @" + // Manual gives -1.7320516
"f * / 3 = f 1                                                   : 0.8660253 @" +  // Manual gives 0.866025
"f * / 6 = f 3                                                   : 0.5773501 @" +  // Manual gives 0.57735
"d .5 f 0 1                                                      : 30. @" +        // p32-33
".75 f 0 2                                                       : 41.409622 @" +  // Manual gives 41.40962
"1 s f 0 3                                                       : -45. @" +       // Manual give -44.99999
"2 s f / f 0 1                                                   : -30. @" +
"4 s f / f 0 2                                                   : 104.47751 @" +  // Manual gives 104.4775
"6 f / f 0 3                                                    d: 9.4623184 @" +  // Manual gives 9.462318  
".5 f 0 2                                                        : 1.0471975 @" +  // Manual gives 1.047197
"99999999 f 0 3                                                  : 1.5707963 @" +  // Manual gives 1.570796
"258963 * =                                                      : C670.61835 @" + // p34-35
"c c f s 4785 f . 5896 f . 7458 f . c f =                        : 1.1328080 @" +
"987456 * 36250 / c 3.2 =                                        : 111.86025 @" +
"2 * 3 * 5 c 4 =                                                 : 24. @" +        // p36-37
"123.45678 f f c 9 / 7 =                                         : 17.636684 @" +
"d 12 3 f f 3                                                   d: 0.2125565 @" +  // Manual gives 0.212556
"22 / 7 = 8 f f 8                                                : 0.4973246 @" +  // Manual gives 0.497325
"3 f 4 - f / / 2 =                                               : 10.017874 @" +  // p40-41 Manual gives 10.017871
"3 s f 4 + f / / 2 =                                             : 10.067675 @" +
"3 f / * 2 = f 4 + 1 / 2 i - 1 i =                               : 0.3215128 @" +  // Manual gives 0.3215125 
"5 s f s * + 1 = f 9 + f = = f 7                                 : -2.3124385 @" + // Manual gives -2.312438
".75 - 1 i / 2 i - 1 = f 9 f 7                                   : 0.972955 @" +   // Manual gives 0.972954
"180 / f * = / = 10 s =                                          : -0.174533 @" +  // p42-43 Manual gives -0.1745329
"75 =                                                            : 1.3089969 @" +
"257.83101 =                                                     : 4.5 @" +
"180 / f * = * = 1 =                                             : 57.29578 @" +
"6 =                                                             : 343.77468 @" +
"3.4906585 s =                                                   : -200.00001 @" + // Manual gives -200
"d c f s 4 f . / 3 f . i = f 0 3                                 : 36.869897 @" +  // p44-45 Manual gives 38.869897 (mistake?)
"f = f 9                                                        d: 5. @" +
"f * / 3 = f s f 2 * 100 =                                       : 50. @" +
"i f i f 1 * f i =                                               : 86.60253 @" +   // Manual gives 86.6025
"c f s 12 f . + 13 f. + 14 f . + 15 f . + 16 f . = / 5 *         : 14. @" +        // p46-47
"i = = f - f = / 4 = f 9                                         : 1.5811388 @" +
"f * / 12 = f s * 5 = f 4 f i * 7 = f 1 * f = =                  : 3.5762981 @" +  // p48-49 Manual gives 3.5762975
"2 f 9 * 7 = f 1 * 5 * 2 f 9 = f 4                               : 0.0394749 @" +  // Manual gives 0.039475
"2 f / f s 8 f / f + 30 f / f + f = f /                          : 1.5189874 @" +  // p50-51
"f * / 3 = f s f 1 f 6 .6666666 = f i f 2 f 6 .6666666 = f + f = : 1.5385207 @" +  // Manual gives 1.538522
"c f s 4 f . 3 f . f = f 9                                       : 5. @" +         // p52-53
"12 f . f = f 9                                                  : 13. @" +
"84 f . f = f 9                                                  : 85. @" +
"c f s 8 f . 15 f . 144 f . f = f 9                              : 145. @" +
"d 21 * = s f s 2 * 12 f . * 13 f . / f = i = f 0 2              : 114.22094 @" +  // p54-55 Manual gives 114.2209
"f 1 * 13 / 21 = f 0 1                                           : 34.371249 @" +  // Manual gives 34.37129
"c f s 50 f 2 * 12 f . * 13 f . * 2 = f - f = f 9                : 10.604257 @" +  // p56-57 Manual gives 10.604251
"50 f 1 * 13 / f = f 9 = f 0 1                                   : 69.902825 @" +  // Manual gives 69.90285
"- 69                                                            : 69. @" +
"* 60 - 54                                                       : 54. @" +
"* 60 =                                                          : 10.17 @" +      // Manual gives 10.26
"53 + 68 = f 1 / 84 i = f s 68 f 1 * f = =                       : 90.86142 @" +   // p58-59 Manual gives 90.861472
"53 f 1 * f = =                                                  : 78.264046 @" +  // Manual gives 78.264024

"oo:0.";                                                 // last line

// ---------------------------------------------------------
function test() {

    var answer = false;                 // State indicating reading expected answer
    var testnum = 0;                    // test number
    var failures = new Array();         // List of test failures
    var fidx = 0;                       // Failure index
    var testchar = "";                  // Current input character

    if (!confirm("Run the self tests?\n(may take some seconds)"))
        return;

    testing = true;

    if (!batt_on)
        sw_batt();

    if (deg_on)
        sw_deg();

    // Scan through all test data ...
    for (tidx = 0; tidx < testdata.length; tidx = tidx + 1) {

        // get next input character
        testchar = testdata.charAt(tidx);

        // Non white-space input
        if (testchar != ' ') {
            // If answer delimiter, clear expected answer string and set state
            if (testchar == ':') {
                answer = true;
                expected = "";
            }
            // If test delimiter, check answer
            else if (testchar == '@') {
                // Clear state
                answer = false;

                // Get calculator's result
                result = disp;

                // For an over/under flow error, check sign match
                if (expected.charAt(0) == 'C') {
                    if (!error)
                        failures[fidx++] = testnum;

                    // Make answer and result sign independent before checking
                    // actual value
                    expected = expected.substring(1, expected.length);
                    if (result.charAt(0) == '-')
                        result = result.substring(1, result.length);
                }

                // Check result matches expected, and log if not
                if (parseFloat(expected) != parseFloat(result)) 
                    failures[fidx++] = testnum;
 
                testnum++;
            }
            // Whilst in answer state, append input to expected variable
            else if (answer == true)
                expected = expected + testchar;
            // Input key sequences
            else {

                // Abort if an invalid character found in the test data
                if (validkeys.indexOf(testchar) == NOMATCH) {
                    alert("Invalid input character ("+testchar+") in test sequence\n"+
                          "at line "+testnum);
                    return;
                }

                if (testchar == 'o')
                    sw_batt();
                else if (testchar == 'd')
                    sw_deg();
                else
                    key_pressed(testchar);
            }
        }
    }

    // Clear testchar and reuse for failure message
    testchar = "";

    // Add all failure test numbers to string
    for (tidx = 0; tidx < fidx; tidx++) 
        testchar = testchar + " " + failures[tidx];

    // Display test results
    if (fidx > 0)
        alert (fidx+" failures at lines:"+testchar);
    else
        alert ("All "+ (testnum-1) + " tests pass");

    testing = false;
}

