//=======================================================================
//                                                                       
// codec.cpp                                             date: 2002/07/04  
//                                                                       
// Author: Simon Southwell                                               
//                                                                       
// Copyright (c) 2002 Simon Southwell                                                                     
//                                                                       
// $Revision: 1.1 $
//                                                                      
//=======================================================================

//=======================================================================
//                                                                       
// This class implements the LZW algorithm. It has a dictionary space of 
// 4K---See dict.cpp.                       
//                                                                       
//=======================================================================

#include "codec.h"

//=======================================================================
// Constructor & Destructor                                              
//=======================================================================

// Only a default constructor, as all the configuration is done from
// the user command line options in methods set_user_config()

CodecClass::CodecClass(char *errbuf)
{
    compress_mode = TRUE;
    previous_codeword = NULLCW;
    graph = FALSE;
    formatted_output = FALSE;
    printable = FALSE;
    dict_dump = FALSE;
    config_max_str_len = MAXWORDLENGTH;
    ip_filename = "";
    op_filename = "";
    errmsg = errbuf;
}

CodecClass::~CodecClass()
{
}

//=======================================================================
// Run the codec                                                         
//=======================================================================

int CodecClass::run(int argc, char **argv)
{
    int status = NOERROR;
    
    if(status = set_user_config(argc, argv))
        return(status);

    // Create a dictionary (inform whether compressing or decompressing---
    // dictionary is used as CAM in compression, SRAM in decompression)
    dptr = new DictClass(compress_mode);

    // Select compression/decompression routines as specified 
    if (compress_mode) {

        // Create a packer (arguments configure formatter)
        pptr = new PackerClass(compress_mode, ofp, formatted_output, printable);

        // Create a compression encoder
        cptr = new CompClass(config_max_str_len, ifp, graph, graph_file);

        // Connect dictionary and packer, and start compressing from input stream
        cptr->compress(dptr, pptr);

        // If -D option specified, dump the dictionary entries,
        // formatted sensibly, to stderr 
        if(dict_dump) 
            if(status = dptr->dump_dictionary(errmsg))
                return(status);

    } else {
        // Create an unpacker (arguments configure formatter)
        uptr = new UnpackerClass(compress_mode, ifp, ofp, formatted_output, printable);

        // Create a decompression decoder
        dcptr = new DecompClass(config_max_str_len, ofp, errmsg);

        // Connect dictionary and unpacker, and start decompressing from input stream
        if(status = dcptr->decompress(dptr, uptr))
            return(status);

        // If -D option specified, dump the dictionary entries,
        // formatted sensibly, to stderr 
        if(dict_dump) 
            if(status = dptr->dump_dictionary(errmsg))
                return(status);

    }

    if(formatted_output) {
        putc(NEWLINE, ofp);
        putc(NEWLINE, ofp);
    }

    return(status);
}

//=======================================================================
// Configure the codec from the command line options                     
//=======================================================================

int CodecClass::set_user_config(int argc, char **argv)
{
    // Program scratch variables---not part of algorithm 
    int option, i;
    char option_str[DEFAULTSTRSIZE], *str;

    // Define the common command line option argument. 
    sprintf(option_str, "%s", "i:o:PFDgdh");

    // Process the command line options 
    while ((option = getopt(argc, argv, option_str)) != EOF)
        switch(option) {
            // Request to dump dictionary to stderr on completion 
            case 'D':
                dict_dump = TRUE;
                break;
            // Xgraph output 
            case 'g':
                graph = TRUE;
                break;
            // Decompression mode 
            case 'd':
                compress_mode = FALSE;
                break;
            // Selected formatted output in printable format 
            case 'P':
                formatted_output = TRUE;
                printable = TRUE;
                break;

            // Selected formatted output in xterm format 
            case 'F':
                formatted_output = TRUE;
                printable = FALSE;
                break;

            case 'i':
                ip_filename = optarg;
                break;

            case 'o':
                op_filename = optarg;
                break;

            // Unknown option specified 
            case 'h':
            case '?':
                str = argv[0];
                for(i = strlen(argv[0]) - 1; i >= 0; i--) 
                    if(argv[0][i] == '\\' || argv[0][i] == ':') {
                        str = &argv[0][i+1];
                        break;
                    }
                
                // an unrecognised option was specified 
                if(errmsg != NULL)
                    sprintf(errmsg, "Usage: %s [-h] [-d] [-D] [-F|-P] [-g]\n" 
                                    "             [-i <filename>] [-o <filename>]\n"
                                    "\nOptions:\n"
                                    "   -h Print help message\n"
                                    "   -d Perform decompression\n"
                                    "   -i Specify input file\n"
                                    "   -o Specify output file\n"
                                    "   -F Formatted output\n"
                                    "   -P As -F, but in printable format\n"
                                    "   -D Dump dictionary afterwards\n"
                                    "   -g Output statistics using xgraph format\n"
                                    "\n"
                                    "All debug information sent to standard error\n", str);
                return(USER_ERROR);
                break;
        }// end switch 

    // Open files if specified, else use standard IO
    if(strlen(ip_filename) == 0)
        ifp = stdin;
    else {
        if((ifp = fopen(ip_filename, "rb")) == NULL) {
            if(errmsg != NULL)
                sprintf(errmsg, "Error. Unable to open input file for reading\n");
            return(USER_ERROR);
        }
    }

    if(strlen(op_filename) == 0)
        ofp = stdout;
    else {
        if((ofp = fopen(op_filename, "wb")) == NULL) {
            if(errmsg != NULL)
                sprintf(errmsg, "Error. Unable to open output file for writing\n");
            return(USER_ERROR);
        }
    }

    // Output the Xgraph header if generating a graph 
    if (compress_mode && graph) {
        if((graph_file = fopen(GRAPHFILENAME,"w")) == NULL) {
            if(errmsg != NULL)
                sprintf(errmsg, "Error. Unable to open graph file for writing\n");
            return(USER_ERROR);
        }
        fprintf(graph_file, "TitleText: LZW C-model performance\n");
        fprintf(graph_file, "%8d %8f\n", 0, 0.0);
    }
    return(NOERROR);
}

