//=======================================================================
//                                                                       
// compress.c                                            date: 2002/07/04  
//                                                                       
// Author: Simon Southwell                                               
//                                                                       
// Copyright (c) 2002 Simon Southwell                                                                     
//                                                                       
// $Revision: 1.1 $
//                                                                      
//=======================================================================

//=======================================================================
// The comp class implements the LZW compression algorithm. It takes a
// stream of bytes from the input, and sends 12 bit codeword to the
// packer, using the dictionary to store the entries for previously seen
// strings.
//=======================================================================

#include "comp.h"

//=======================================================================
// Constructor and Destructor                                            
//=======================================================================

// Default constructor
CompClass::CompClass()
{
    match_length_so_far = 0;
    set_max_string_length(MAXWORDLENGTH);
    set_graph(FALSE, (FILE *)NULL);
    ip_file = stdin;
    ip_bytecount = 0;
    op_bytecount = 0;
}

// Constructor with configuration parameters
CompClass::CompClass(int maxstrlen, FILE * ifp, bool grph, FILE *fp)
{
    match_length_so_far = 0;
    set_max_string_length(maxstrlen);
    set_graph(grph, fp);
    ip_file = ifp;
    ip_bytecount = 0;
    op_bytecount = 0;
}

CompClass::~CompClass()
{
}

//========================================================================
//                                                                        
// Function name: compress                                                
//                                                                        
// Description: 
//    Performs LZW compression on standard input outputing codewords 
//    (via the packer) to standard output.                                                                 
//                                                                        
//========================================================================

void CompClass::compress(DictClass *dptr, PackerClass *pptr)
{

    previous_codeword = NULLCW;
    match_length_so_far = 0;
    code_size = dptr->reset_dictionary();

    // Process bytes for the while length of the file.
    while ((ipbyte = getc(ip_file)) != EOF) {

        output_graphics_data_point();

        // Increment the byte counter for each input 
        ip_bytecount++;

        // First byte, so we need to go round the loop once more for
	// another byte, and find the root codeword representation for 
	// this byte.  
        if (previous_codeword == NULLCW) {

            previous_codeword = convert_to_rootcw(ipbyte);

            // We have an implied root codeword match i.e. match length = 1 
            match_length_so_far = 1;

        // Otherwise, process the string as normal 
        } else {
             
            // Match found 
            if ((match_addr = dptr->entry_match(previous_codeword, (unsigned char)ipbyte)) != NOMATCH) {

                // A match increases our string length representation by
                // one. This is used simply to check that we can fit on
		// the stack at decompression (shouldn't reach this 
		// limit).
                match_length_so_far++;

                // Previous matched codeword becomes codeword value of dictionary 
		// entry we've just matched 
                previous_codeword = match_addr;

            // Match not found 
            } else { // entry_match(addr) is TRUE 

                // Output the last matched codeword 
		op_bytecount += pptr->pack(previous_codeword, code_size);

                // Build an entry for the new string (if possible) 
                code_size = dptr->build_entry(previous_codeword, (unsigned char)ipbyte);
                
                // Carry forward the input byte as a 'matched' root codeword 
                previous_codeword = convert_to_rootcw(ipbyte);

                // Now we have just a single root codeword match, yet to be processed
                match_length_so_far = 1;

            }

        } // endelse (previous_codeword == NULLCW) 

    } // end while 

    // If we've terminated and still have a codeword to output, 
    // then we have to output the codeword which represents all the 
    // matched  string so far (and it could be just a root codeword).
    if (previous_codeword != NULLCW) {
	op_bytecount += pptr->pack(previous_codeword, code_size);

        // Pipeline flushed, so no previous codeword 
        previous_codeword = NULLCW;
        match_length_so_far = 0;
    }

    // We let the packer know we've finished and thus to flush its pipeline 
    op_bytecount += pptr->pack(EOFFLUSH, code_size);

} // end compress() 

//=======================================================================
//                                                                       
// Function name: output_graphics_data_point                             
//                                                                       
// Description: 
//    Outputs a compression ratio every DATA_POINT_PERIOD
//    into the file pointed to by graph_file. The format is    
//    Xgraph compatible.                                       
//                                                                       
//=======================================================================

void CompClass::output_graphics_data_point(void)
{
    static int last_ipcount = 0, last_opcount = 0;
 
    // Output a graphic data point for every 'n' input bytes if the
    // graph option has been chosen 
    if (graph && (ip_bytecount > 0) && (ip_bytecount%DATA_POINT_PERIOD == 0)) {
        fprintf(graph_file, "%8d %8f\n",
                ip_bytecount,
                1.0 * (ip_bytecount - last_ipcount) /
                (op_bytecount - last_opcount)
                );
        last_ipcount = ip_bytecount;
        last_opcount = op_bytecount;
    }
}

