//=======================================================================
//                                                                        
// decomp.cpp                                            date: 2002/07/04          
//                                                                        
// Author: Simon Southwell                                                
//                                                                        
// Copyright (c) 2002 Simon Southwell                                                                     
//                                                                       
// $Revision: 1.1 $
//                                                                      
//=======================================================================

//=======================================================================
//  The decomp class takes a stream of valid LZW codewords (through  
//  calls to the unpacker) and formulates a dictionary as it goes. The  
//  output is a byte sequence constructed  from following a linked list  
//  of entries starting with the entry for the input codeword until  
//  reaching a root  codeword (pointing to NULL).The bytes are pushed  
//  onto a stack as the list is followed, and then flushed upon list  
//  termination. An exception condition  exists (called the K omega K  
//  case, as originally  described by Welch) where a codeword is input  
//  that has a yet to be constructed entry in the linked list it points
//  to. The code will reconstruct the entry before proceeding down the
//  rest of the list.                                                                                                                                                                
//=======================================================================

#include "decomp.h"

//=======================================================================
// Constructors and Destructor                                           
//=======================================================================

DecompClass::DecompClass()
{
    max_string_length = MAXWORDLENGTH;
    op_bytecount = 0;
    ip_bytecount = 0;
    op_file = stdout;
    code_size = MINCWLEN;
    previous_codeword = NULLCW;
    reset_stack();
    errmsg = NULL;
}

DecompClass::DecompClass(int maxstr, FILE *ofp, char *errbuf)
{
    max_string_length = maxstr;
    op_bytecount = 0;
    ip_bytecount = 0;
    op_file = ofp;
    code_size = MINCWLEN;
    previous_codeword = NULLCW;
    reset_stack();
    errmsg = errbuf;
}

DecompClass::~DecompClass()
{
}

//========================================================================
//                                                                        
// Function name: decompress                                              
//                                                                        
// Description: 
//    Performs LZW decompression of codewords from            
//    standard input outputing decompressed data to standard    
//    output.                                                   
//                                                                        
//========================================================================

int DecompClass::decompress(DictClass *dptr, UnpackerClass *uptr)
{
    unsigned int byte_count;
    int status = NOERROR;

    // Keep going until thare are no more codewords 
    while (byte_count = uptr->unpack(&ip_codeword, code_size)) {

        ip_bytecount += byte_count;

        if (dptr->codeword_valid(ip_codeword)) {

            // Traverse down the dictionary's linked list placing bytes onto 
            // the stack. Empty the stack when reached a NULLCW pointer and remember 
            // the last flushed  byte.
            string_terminator_byte = output_linked_list(dptr, uptr, &status);

            if(status)
                return(status);

            // We must build a dictionary entry using the last codeword fully 
            // processed and the first flushed byte of the present codeword (if we 
            // aren't flushed)
            if (previous_codeword != NULLCW)
                code_size = dptr->build_entry(previous_codeword, string_terminator_byte);

        } else {  // No valid entry exists
            // Unknown codeword error
            if(errmsg != NULL)
                sprintf(errmsg, "***decompress: Error --- UNKNOWN CODEWORD (0x%08x)\n", ip_codeword);
            return(DECOMPRESSION_ERROR);
        }

        previous_codeword = ip_codeword;

    }// end while 

    return(status);
}

//========================================================================
//                                                                        
// Function name: output_linked_list                                      
//                                                                        
// Description: 
//    Follows a linked list of dictionary entries and outputs
//    the bytes in reverse order.                  
//                                                                        
//========================================================================

char DecompClass::output_linked_list(DictClass *dptr, UnpackerClass *uptr, int *errflag)
{
    pointer_type pointer;
    byte_type byte;

    errflag = FALSE;

    pointer = ip_codeword;

    // While not at the end of the list, follow the linked list, placing
    // byte values on a stack 
    while (pointer != NULLCW) {

        // If not a root codeword ... 
        if (!dptr->root_codeword(pointer)) {

            // If an entry in the linked list is the next free codeword,
            // then it must need building as a KwK case. 
            if(dptr->is_next_free_entry(pointer) && (previous_codeword != NULLCW)) {

                // The pointer and byt values are as for the KwK build;
                // i.e. the last codeword that was input and its first
                // character 
                byte = string_terminator_byte;
                pointer = previous_codeword;

            } else {

                // Get the byte and pointer values from the entry. 
                byte = dptr->dictionary_entry_byte(pointer);
                pointer = dptr->dictionary_entry_pointer(pointer);

            }

	// We have to generate the entry for root codewords 
        } else {
	    byte = pointer;
            pointer = NULLCW;
        }

        // It is an error to have a codeword which overflows the stack 
        if (stack_size() == (unsigned int)max_string_length) {
            if(errmsg != NULL)
                sprintf(errmsg, "decompress: Error --- BAD WORD LENGTH\n");
            *errflag = DECOMPRESSION_ERROR;
            return(NULL);
        }

        // Push the current byte value on the stack 
        push(byte);

    } // end while 

    // Remember the length of the codeword string so that it can be
    // capped within the main decompression algorithm 
    last_word_length = stack_size();

    // Now flush the stack to the output stream 
    do {
        if(uptr->is_formatted_output())
            pop();
        else
            putc(pop(), op_file);
        op_bytecount++;
    }
    while (!stack_empty());

    // Return the last flushed byte, for use in building dictionary entries
    return(byte);
}

