//=======================================================================
//                                                                       
// dict.cpp                                              date: 2002/07/04  
//                                                                       
// Authors: Simon Southwell                                              
//                                                                       
// Copyright (c) 2002 Simon Southwell                                                                     
//                                                                       
// $Revision: 1.1 $
//                                                                      
//=======================================================================

//=======================================================================
// Dictionary access utilities for LZW algorithm                       
//                                                                       
// The dictionary is implemented as a contiguous array of a DICTIONARY   
// structure (dictionary[]) with a 12 bit pointer (for 4K dictionary)    
// space and a 'byte' value. 
//                                                                       
//=======================================================================

#include "dict.h"

//=======================================================================
// Constructor                                                           
//=======================================================================
// Default
DictClass::DictClass()
{
    next_available_codeword = FIRSTCW;
    compress_mode = TRUE;
    codeword_len = MINCWLEN;
}

// Specify compress/decompress mode
DictClass::DictClass(bool mode)
{
    next_available_codeword = FIRSTCW;
    compress_mode = mode;
    codeword_len = MINCWLEN;
}

//=======================================================================
// Destructor                                                            
//=======================================================================
DictClass::~DictClass()
{
}

//=======================================================================
// Function name: reset_dictionary                                       
//                                                                       
// Description: 
//    Resets the dictionary
//=======================================================================

unsigned int DictClass::reset_dictionary()
{
    // Reset common state 
    next_available_codeword = FIRSTCW;

    return(MINCWLEN);
}

//=======================================================================
//                                                                       
// Function name: entry_match                                            
//                                                                       
// Description: 
//    Returns TRUE if the dictionary entry at the specified
//    address matches the specified byte. Otherwise FALSE.                   
//                                                                                                   
//=======================================================================

dictionary_address_type DictClass::entry_match(pointer_type pointer, 
                                          byte_type byte)
{
    dictionary_address_type addr;

    // Get possible dictionary entry value for match. 
    // (This is not part of algorithm, but a method for 
    // speeding up dictionary searches).
    addr = indirection_table[pointer][byte];

    // Test to see if we have a match at the address, and is in the
    // valid portion of the dictionary. 
    if (!(addr >= FIRSTCW && addr < next_available_codeword &&
		(dictionary_entry_byte(addr) == byte) &&
         (dictionary_entry_pointer(addr) == pointer)))
        addr = NOMATCH; // Set addr to indicate no match 
        
    // Return address of match (or no match) 
    return ((dictionary_address_type) addr);
}

//=======================================================================
//                                                                       
// Function name: build_entry                                            
//                                                                       
// Description: 
//    Creates a new dictionary entry at next_free_code, so
//    long as the dictionary isn't full, in which case the     
//    build is not performed, and a partial_reset is done      
//    instead.            
//                                                                       
//=======================================================================

unsigned int DictClass::build_entry(codeword_type codeword, 
                               byte_type     byte)
{

    // If the dictionary is full, reset it before doing a build
    if(dictionary_full()) 
        codeword_len = reset_dictionary();

    // Set the entry values for the pointer and bytes 
    set_dictionary_entry_pointer(next_available_codeword, codeword);
    set_dictionary_entry_byte(next_available_codeword, byte);

    // Set the pointer table to point into the dictionary. (This is not
    // part of the algorithm, but a mechanism for fast dictionary
    // accesses.)  
    if(compress_mode)
       indirection_table[codeword][byte] = next_available_codeword;

    // If we've just built an entry whose codeword value is greater than 
    // the current output codeword size, then increment the current codeword 
    // size. The decompression builds lag behind by one, so this event
    // is anticipated by an entry. 
    if(codeword_len < MAXCWLEN) {
        if(next_available_codeword == (unsigned)(1 << codeword_len) - (!compress_mode ? 1 : 0))
            codeword_len++;

    // If decompressing and we're one short of having a full dictionary,
    // reset the codeword_len. This anticipates a reset next build,
    // in a similar way as for the codeword length increment. 
    } else if(!compress_mode && next_available_codeword == DICTFULL-1)  
        codeword_len = MINCWLEN;

    if (next_available_codeword != DICTFULL)
        ++next_available_codeword;

    return(codeword_len);
}

//=======================================================================
//                                                                       
// Function name: dump_dictionary                                        
//                                                                       
// Description: 
//    Dumps the dictionary to stderr in a readable form
//                                                                       
//=======================================================================

int DictClass::dump_dictionary(char *errmsg)
{
    FILE *dfp;
    dictionary_address_type address;

    if((dfp = fopen(DICTDUMPFNAME, "wb")) == NULL) {
        if(errmsg != NULL)
            sprintf(errmsg, "***Error: couldn't open file %s for writing. Exiting.\n", DICTDUMPFNAME);
        return(USER_ERROR);
    }

    fprintf(dfp, "\nNext Available Codeword at 0x%04x\n", next_available_codeword);
    fprintf(dfp, "\nAddress   pointer byte\n");
    fprintf(dfp, "----------------------\n");
    for (address = FIRSTCW; address < next_available_codeword; address++) {
            fprintf(dfp, " 0x%04x    0x%03x  0x%02x\n",
                    address,
                    dictionary_entry_pointer(address),
                    dictionary_entry_byte(address));
    }

    fclose(dfp);

    return (NOERROR);
}

