//=======================================================================
//                                                                       
// unpack.cpp                                            date: 2002/07/04  
//                                                                       
// Author: Simon Southwell                                               
//
// Copyright (c) 2002 Simon Southwell                                                                     
//                                                                       
// $Revision: 1.1 $
//                                                                      
//=======================================================================

//=======================================================================
// The unpack class takes the variable sized codewords from the input
// file stream and constructs fixed size codewords suitable for the 
// decompress object, returning them via a passed pointer. The current
// length of the input codewords is also passed in, as this is 
// determined by the dictionary.
//=======================================================================

#include "unpack.h"

//=======================================================================
// Constructor and Destructor                                            
//=======================================================================

UnpackerClass::UnpackerClass()
{
    set_formatter_mode(FALSE, FALSE, TRUE);
    currlen = 0;
    barrel= 0;
    op_codeword = 0;
    ip_file = stdin;
    op_file = stdout;
}

UnpackerClass::UnpackerClass(bool compmode, FILE *ifp, FILE *ofp, bool formop, bool prnt)
{
    set_formatter_mode(compmode, formop, prnt);
    currlen = 0;
    barrel= 0;
    op_codeword = 0;
    ip_file = ifp;
    op_file = ofp;
}

UnpackerClass::~UnpackerClass()
{
}

//=======================================================================
//                                                                       
// Function name: unpack                                                 
//                                                                       
// Description:                                                          
//    unpack() grabs bytes from input stream, placing then on a barrel
//    shifter until it has enough bits for a codeword of the current 
//    codeword length (codeword_length). 
//                                                                       
//=======================================================================

unsigned int UnpackerClass::unpack(codeword_type *codeword, unsigned int codeword_length)
{
    int byte_count = 0;

    // Start inputing bytes to form a whole codeword 
    do {
        // Gracefully fail if no more input bytes---codeword is
        // don't care. 
        if ((ipbyte = getc(ip_file)) == EOF) {
            return(byte_count);
        }

        // We successfully got a byte so increment the byte counter 
        byte_count++;

        // Put the byte on the barrel shifter 
        barrel |= (ipbyte & BYTEMASK) << currlen;

        // We have another byte's worth of bits 
        currlen += BYTESIZE;

    // Continue until we have enough bits for a codeword.
    } while (currlen < codeword_length);

    // Codeword is bottom 'codeword_length' bits: I.e. mask=2^codeword_length - 1 
    op_codeword = (barrel & ((0x1 << codeword_length) - 1));
    currlen -= codeword_length;
    barrel >>= codeword_length;

    // Output codeword to user display formatter 
    format(op_codeword, op_file);

    // Return the codeword value in the pointer 
    *codeword = op_codeword;

    // Mark the operation as successful 
    return(byte_count);
}

